unit CompressedDIB;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  TurboSprite;

type

  PByte = ^byte;

  TCompressedDIB = class( TDIB )
  private
    MemoryNeeded: integer;
    Savings: single;
    xDummy: single;
    nDummy: integer;
  protected
    procedure ReadImage( Stream: TStream ); override;
    procedure WriteImage( Stream: TStream ); override;
  public
    procedure LoadFromFile( const s: TFileName ); override;
  published
    property Compression: single read Savings write xDummy stored false;
    property Size: integer read MemoryNeeded write nDummy stored false;
  end;

procedure Register;

implementation

procedure TCompressedDIB.LoadFromFile( const s: TFileName );
var
  strRuns: TStrings;
  Location: integer;
  p: PByte;
  CurrentByte, Run: byte;
  i: integer;
  Value, Number: byte;
  sLine: string;
  nPos: integer;
begin
  inherited LoadFromFile( s );
{ Now that we have to physical image data, we'll begin constructing strings
  that represent the "runs", and increment a variable that tracks the total
  memory we'll need for the "run" information }
  strRuns := TStringList.Create;
  Location := 0;
  Run := 0;
  p := pDIBBits;
  while ( Location < DIBCanvas.Size ) do
  begin
    if (Location > 0) then
    begin
      if (p^ <> CurrentByte) or (Run = 255) then
      begin
        strRuns.Add( IntToStr( Run ) + ':' + IntToStr( CurrentByte ) );
        Run := 0;
      end
      else
        Inc( Run );
    end;
    CurrentByte := p^;
    Inc( p );
    Inc( Location );
  end;
  if Run > 0 then
    strRuns.Add( IntToStr( Run ) + ':' + IntToStr( CurrentByte ) );
{ We now know how much memory we need }
  MemoryNeeded := strRuns.Count * 2;
{ Free the old buffer, and allocate a new one }
  if DIBCanvas.Size > 0 then
    Savings := 100.0 - (MemoryNeeded / DIBCanvas.Size * 100);
  CleanUp;
  pDIBBits := VirtualAlloc( nil, MemoryNeeded, MEM_RESERVE or MEM_COMMIT, PAGE_READWRITE );
{ Parse the strings, and copy bytes into the buffer }
  p := pDIBBits;
  for i := 0 to strRuns.Count - 1 do
  begin
    sLine := strRuns[i];
    nPos := Pos( ':', sLine );
    Number := StrToInt( Copy(sLine, 1, nPos - 1) );
    Value := StrToInt( Copy(sLine, nPos + 1, Length( sLine )) );
    p^ := Number;
    Inc( p );
    p^ := Value;
    Inc( p );
  end;
  strRuns.Free;
end;

procedure TCompressedDIB.ReadImage( Stream: TStream );
begin
  CleanUp;
  Stream.Read( MemoryNeeded, SizeOf( integer ) );
  Stream.Read( Savings, SizeOf( single ) );
  if MemoryNeeded > 0 then
  begin
    pDIBBits := VirtualAlloc( nil, MemoryNeeded, MEM_RESERVE or MEM_COMMIT, PAGE_READWRITE );
    Stream.Read( pDIBBits^, MemoryNeeded );
  end;
end;

procedure TCompressedDIB.WriteImage( Stream: TStream );
begin
  Stream.Write( MemoryNeeded, SizeOf( integer ) );
  Stream.Write( Savings, SizeOf( single ) );
  if (MemoryNeeded) > 0 then
     Stream.Write( pDIBBits^, MemoryNeeded );
end;

procedure Register;
begin
  RegisterComponents( 'TurboSprite', [TCompressedDIB] );
end;

end.
